<?php
use Livewire\Volt\Component;
use App\Models\FastEvent;
use Carbon\Carbon;

new class extends Component {
    // Main Info
    public $fastEvents = [];
    public $showModal = false;
    public $editingEvent = null;
    public $eventForm = [
        'event_name' => '',
        'is_active' => true,
        'expired_at' => '',
        'blade_name' => ''
    ];

    /**
     * تهيئة البيانات عند تحميل الكومبوننت
     */
    public function mount()
    {
        $this->loadData();
    }

    public function loadData()
    {
        try {
            $events = FastEvent::orderBy('created_at', 'desc')->get();
            $this->fastEvents = $events->isNotEmpty() ? $events->toArray() : [];
        } catch (\Exception $e) {
            $this->fastEvents = [];
            session()->flash('error', 'Error loading events: ' . $e->getMessage());
        }
    }

    /**
     * تحديث البيانات
     */
    public function refreshData()
    {
        $this->loadData();
    }

    /**
     * فتح نافذة إنشاء حدث جديد
     */
    public function openCreateModal()
    {
        $this->resetForm();
        $this->showModal = true;
    }

    /**
     * فتح نافذة تعديل الحدث
     */
    public function openEditModal($eventId)
    {
        $event = FastEvent::find($eventId);
        if ($event) {
            $this->editingEvent = $event->id;
            $this->eventForm = [
                'event_name' => $event->event_name,
                'is_active' => $event->is_active,
                'expired_at' => $event->expired_at ? $event->expired_at->format('Y-m-d\TH:i') : '',
                'blade_name' => $event->blade_name
            ];
            $this->showModal = true;
        }
    }

    /**
     * حفظ الحدث (إنشاء أو تعديل)
     */
    public function saveEvent()
    {
        $this->validate([
            'eventForm.event_name' => 'required|string|max:255',
            'eventForm.expired_at' => 'nullable|date|after:now',
            'eventForm.blade_name' => 'nullable|string|max:255'
        ]);

        try {
            if ($this->editingEvent) {
                $event = FastEvent::find($this->editingEvent);
                $event->update([
                    'event_name' => $this->eventForm['event_name'],
                    'is_active' => $this->eventForm['is_active'],
                    'expired_at' => $this->eventForm['expired_at'] ? Carbon::parse($this->eventForm['expired_at']) : null,
                    'blade_name' => $this->eventForm['blade_name']
                ]);
            } else {
                FastEvent::create([
                    'event_name' => $this->eventForm['event_name'],
                    'is_active' => $this->eventForm['is_active'],
                    'expired_at' => $this->eventForm['expired_at'] ? Carbon::parse($this->eventForm['expired_at']) : null,
                    'blade_name' => $this->eventForm['blade_name']
                ]);
            }

            $this->loadData();
            $this->closeModal();
            session()->flash('message', $this->editingEvent ? 'تم تحديث الحدث بنجاح' : 'تم إنشاء الحدث بنجاح');
        } catch (\Exception $e) {
            session()->flash('error', 'حدث خطأ: ' . $e->getMessage());
        }
    }

    /**
     * حذف الحدث
     */
    public function deleteEvent($eventId)
    {
        try {
            $event = FastEvent::find($eventId);
            if ($event) {
                $event->delete();
                $this->loadData();
                session()->flash('message', 'تم حذف الحدث بنجاح');
            }
        } catch (\Exception $e) {
            session()->flash('error', 'حدث خطأ في الحذف: ' . $e->getMessage());
        }
    }

    /**
     * تغيير حالة الحدث (نشط/غير نشط)
     */
    public function toggleEventStatus($eventId)
    {
        try {
            $event = FastEvent::find($eventId);
            if ($event) {
                $event->update(['is_active' => !$event->is_active]);
                $this->loadData();
                session()->flash('message', 'تم تغيير حالة الحدث بنجاح');
            }
        } catch (\Exception $e) {
            session()->flash('error', 'حدث خطأ في تغيير الحالة: ' . $e->getMessage());
        }
    }

    /**
     * إغلاق النافذة المنبثقة
     */
    public function closeModal()
    {
        $this->showModal = false;
        $this->resetForm();
    }

    /**
     * إعادة تعيين النموذج
     */
    private function resetForm()
    {
        $this->editingEvent = null;
        $this->eventForm = [
            'event_name' => '',
            'is_active' => true,
            'expired_at' => '',
            'blade_name' => ''
        ];
    }
};
?>
<div>
    <div class="space-y-8">

        <!-- Page Header -->
        <div class="text-center">
            <div
                class="w-16 h-16 mx-auto mb-4 bg-gradient-to-r from-indigo-500 to-purple-600 rounded-2xl flex items-center justify-center">
                <i class="fas fa-gift text-white text-2xl"></i>
            </div>
            <h2 class="text-3xl font-bold text-gray-900 dark:text-white mb-2">إدارة نظام المكافآت والأحداث السريعة</h2>
            <p class="text-gray-600 dark:text-gray-400">إدارة الأحداث السريعة والمكافآت المتاحة للاعبين</p>
        </div>

        <!-- Error Messages -->
        @if (session()->has('error'))
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
                {{ session('error') }}
            </div>
        @endif

        @if (session()->has('message'))
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded">
                {{ session('message') }}
            </div>
        @endif

        <!-- Fast Events Management Section -->
        <div class="bg-white dark:bg-gray-800 rounded-xl shadow-lg">
            <div class="p-6 border-b border-gray-200 dark:border-gray-700">
                <div class="flex justify-between items-center">
                    <h3 class="text-xl font-semibold text-gray-900 dark:text-white">إدارة الأحداث السريعة</h3>
                    <button wire:click="openCreateModal"
                            class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg transition-colors">
                        <i class="fas fa-plus mr-2"></i>
                        إضافة حدث جديد
                    </button>
                </div>
            </div>

            <div class="p-6">
                @if($fastEvents && count($fastEvents) > 0)
                    <div class="overflow-x-auto">
                        <table class="w-full table-auto">
                            <thead>
                                <tr class="bg-gray-50 dark:bg-gray-700">
                                    <th class="px-4 py-3 text-right text-sm font-medium text-gray-900 dark:text-white">ID</th>
                                    <th class="px-4 py-3 text-right text-sm font-medium text-gray-900 dark:text-white">اسم الحدث</th>
                                    <th class="px-4 py-3 text-center text-sm font-medium text-gray-900 dark:text-white">الحالة</th>
                                    <th class="px-4 py-3 text-center text-sm font-medium text-gray-900 dark:text-white">تاريخ الانتهاء</th>
                                    <th class="px-4 py-3 text-center text-sm font-medium text-gray-900 dark:text-white">ملف Blade</th>
                                    <th class="px-4 py-3 text-center text-sm font-medium text-gray-900 dark:text-white">الإجراءات</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-gray-200 dark:divide-gray-600">
                                @foreach($fastEvents as $event)
                                    <tr class="hover:bg-gray-50 dark:hover:bg-gray-700">
                                        <td class="px-4 py-3 text-sm text-gray-900 dark:text-white">{{ $event->id }}</td>
                                        <td class="px-4 py-3 text-sm text-gray-900 dark:text-white">{{ $event->event_name }}</td>
                                        <td class="px-4 py-3 text-center">
                                            @if($event->is_active && !$event->is_expired)
                                                <span class="px-2 py-1 text-xs font-medium bg-green-100 text-green-800 rounded-full">
                                                    نشط
                                                </span>
                                            @elseif($event->is_expired)
                                                <span class="px-2 py-1 text-xs font-medium bg-red-100 text-red-800 rounded-full">
                                                    منتهي الصلاحية
                                                </span>
                                            @else
                                                <span class="px-2 py-1 text-xs font-medium bg-gray-100 text-gray-800 rounded-full">
                                                    غير نشط
                                                </span>
                                            @endif
                                        </td>
                                        <td class="px-4 py-3 text-center text-sm text-gray-900 dark:text-white">
                                            {{ $event->expired_at ? $event->expired_at->format('Y-m-d H:i') : 'غير محدد' }}
                                        </td>
                                        <td class="px-4 py-3 text-center text-sm text-gray-900 dark:text-white">
                                            {{ $event->blade_name ?: 'غير محدد' }}
                                        </td>
                                        <td class="px-4 py-3 text-center">
                                            <div class="flex justify-center space-x-2 space-x-reverse">
                                                <button wire:click="openEditModal({{ $event->id }})"
                                                        class="text-blue-600 hover:text-blue-800 transition-colors">
                                                    <i class="fas fa-edit"></i>
                                                </button>
                                                <button wire:click="toggleEventStatus({{ $event->id }})"
                                                        class="text-yellow-600 hover:text-yellow-800 transition-colors">
                                                    <i class="fas fa-{{ $event->is_active ? 'pause' : 'play' }}"></i>
                                                </button>
                                                <button wire:click="deleteEvent({{ $event->id }})"
                                                        onclick="return confirm('هل أنت متأكد من حذف هذا الحدث؟')"
                                                        class="text-red-600 hover:text-red-800 transition-colors">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                @endforeach
                            </tbody>
                        </table>
                    </div>
                @else
                    <div class="text-center py-8">
                        <i class="fas fa-calendar-times text-gray-400 text-4xl mb-4"></i>
                        <p class="text-gray-500 dark:text-gray-400">لا توجد أحداث سريعة</p>
                    </div>
                @endif
            </div>
        </div>

        <!-- Modal for Create/Edit Event -->
        @if($showModal)
            <div class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50"
                 wire:click.self="closeModal">
                <div class="bg-white dark:bg-gray-800 rounded-xl p-6 w-full max-w-md mx-4">
                    <div class="flex justify-between items-center mb-4">
                        <h3 class="text-lg font-semibold text-gray-900 dark:text-white">
                            {{ $editingEvent ? 'تعديل الحدث' : 'إضافة حدث جديد' }}
                        </h3>
                        <button wire:click="closeModal" class="text-gray-400 hover:text-gray-600">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>

                    <form wire:submit.prevent="saveEvent" class="space-y-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                اسم الحدث
                            </label>
                            <input type="text" wire:model="eventForm.event_name"
                                   class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-white">
                            @error('eventForm.event_name')
                                <span class="text-red-500 text-xs">{{ $message }}</span>
                            @enderror
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                تاريخ الانتهاء
                            </label>
                            <input type="datetime-local" wire:model="eventForm.expired_at"
                                   class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-white">
                            @error('eventForm.expired_at')
                                <span class="text-red-500 text-xs">{{ $message }}</span>
                            @enderror
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                اسم ملف Blade (اختياري)
                            </label>
                            <input type="text" wire:model="eventForm.blade_name"
                                   placeholder="مثال: daily-bonus"
                                   class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-white">
                            @error('eventForm.blade_name')
                                <span class="text-red-500 text-xs">{{ $message }}</span>
                            @enderror
                        </div>

                        <div class="flex items-center">
                            <input type="checkbox" wire:model="eventForm.is_active"
                                   class="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded">
                            <label class="mr-2 block text-sm text-gray-900 dark:text-white">
                                الحدث نشط
                            </label>
                        </div>

                        <div class="flex justify-end space-x-3 space-x-reverse pt-4">
                            <button type="button" wire:click="closeModal"
                                    class="px-4 py-2 text-gray-600 bg-gray-200 rounded-lg hover:bg-gray-300 transition-colors">
                                إلغاء
                            </button>
                            <button type="submit"
                                    class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
                                {{ $editingEvent ? 'حفظ التغييرات' : 'إضافة الحدث' }}
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        @endif
    </div>
</div>
